import asyncio
from typing import List

import httpx

from cli.config import settings
from cli.internal.wappie.types import (
    Course,
    CourseRegistration,
    GradeWork,
    Submission,
    User,
)

if settings.wappie_api_key == "PLEASE_ENTER_TOKEN":
    raise ValueError(f"missing valid WAPPIE_API_KEY in {settings.Config.env_file}")

DEFAULT_CLIENT_KWARGS = {
    "base_url": settings.wappie_base_url,
    "headers": {"api_key": settings.wappie_api_key},
}


async def get_courses() -> List[Course]:
    async with httpx.AsyncClient(**DEFAULT_CLIENT_KWARGS) as client:
        r = await client.get("/v1/courses")
    return [Course(**c) for c in r.json()]


async def get_course_registrations(
    talentlms_user_id: int = None,
) -> List[CourseRegistration]:
    async with httpx.AsyncClient(**DEFAULT_CLIENT_KWARGS) as client:
        r = await client.get(
            "/v1/course-registrations",
            params={"talentlms_user_id": talentlms_user_id}
            if talentlms_user_id
            else None,
        )
    return [CourseRegistration(**cr) for cr in r.json()]


async def get_grade_work(open: bool) -> List[GradeWork]:
    async with httpx.AsyncClient(**DEFAULT_CLIENT_KWARGS) as client:
        r = await client.get("/v1/grade-work", params={"open": open})
    return [GradeWork(**gw) for gw in r.json()]


async def get_status() -> bool:
    async with httpx.AsyncClient(**DEFAULT_CLIENT_KWARGS) as client:
        r = await client.get("/v1/status")
    return r.json() == "OK"


async def get_submissions(open: bool) -> List[Submission]:
    async with httpx.AsyncClient(**DEFAULT_CLIENT_KWARGS) as client:
        r = await client.get("/v1/submissions", params={"open": open})
    return [Submission(**s) for s in r.json()]


async def get_users(talentlms_user_id: int = None) -> List[User]:
    async with httpx.AsyncClient(**DEFAULT_CLIENT_KWARGS) as client:
        r = await client.get(
            "/v1/users",
            params={"talentlms_user_id": talentlms_user_id}
            if talentlms_user_id
            else None,
        )
    return [User(**u) for u in r.json()]
