from datetime import datetime
import logging

from pydantic import BaseModel, validator
from typing import List, Optional

import cli.internal.talentlms.constants as constants


class TalentLMSBaseModel(BaseModel):
    """Wraps Pydantic BaseModel to provide custom datetime parsing."""

    @validator(
        "created_on",
        "enrolled_on",
        "completed_on",
        "expired_on",
        "last_updated",
        "deactivation_date",
        "creation_date",
        "last_update_on",
        "expiration_datetime",
        pre=True,
        check_fields=False,
    )
    def to_datetime(cls, v):
        try:
            return datetime.strptime(v, constants.date_string_format)
        except:
            return None


class CourseForUser(TalentLMSBaseModel):
    id: str
    name: str
    role: str
    enrolled_on: datetime
    completed_on: Optional[datetime]
    completion_status: str
    completion_percentage: str
    expired_on: Optional[datetime]
    total_time: Optional[str]
    total_time_seconds: int
    last_accessed_unit_url: str


class User(TalentLMSBaseModel):
    id: int
    login: str
    first_name: str
    last_name: str
    email: str
    user_type: str
    timezone: str
    language: str
    status: str
    deactivation_date: Optional[datetime]
    points: str
    created_on: datetime
    last_updated: datetime
    login_key: str
    courses: List[CourseForUser] = []


class CourseUnit(TalentLMSBaseModel):
    id: int
    type: str
    name: str
    url: Optional[str]


class Course(TalentLMSBaseModel):
    id: int
    name: str
    code: str
    category_id: Optional[int]
    description: str
    price: str
    status: str
    creation_date: datetime
    last_update_on: datetime
    creator_id: int
    hide_from_catalog: str
    time_limit: str
    expiration_datetime: Optional[datetime]
    avatar: str
    big_avatar: str
    units: List[CourseUnit] = []


class CourseSubmission(TalentLMSBaseModel):
    user_id: int
    status: str
    score: Optional[int]


class UserProgressUnit(CourseUnit):
    completion_status: str
    completed_on: datetime
    score: str


class UserProgress(TalentLMSBaseModel):
    role: str
    enrolled_on: datetime
    completion_status: str
    completion_percentage: int
    completed_on: datetime
    expired_on: datetime
    total_time_seconds: int
    units: List[UserProgressUnit]
