import json
import logging

import httpx
import typer

from cli.config import settings
from cli.internal.formatting import echo_failure, echo_success
from cli.internal.talentlms import constants
from cli.internal.talentlms.types import Course, CourseSubmission, User, UserProgress


DEFAULT_CLIENT_KWARGS = {
    "headers": {"Content-Type": "application/json"},
    "timeout": 10,
    "base_url": constants.api_base_url,
    "auth": (settings.talent_lms_api_key, ""),
}


async def call_talentlms_api(
    path: str,
    client_kwargs: dict = DEFAULT_CLIENT_KWARGS,
    method: str = "GET",
    params: dict = {},
) -> dict:
    logging.debug(f"{method} {path}")
    try:
        async with httpx.AsyncClient(**client_kwargs) as client:
            if method == "GET":
                r = await client.get(path)
            elif method == "POST":
                r = await client.post(path, data=json.dumps(params))
        j = r.json()
        if "error" in j:
            raise Exception(json.dumps(j["error"]))
        return j
    except Exception as e:
        echo_failure(f"{path}: {e}")
        raise typer.Exit()


async def get_user_by_id(id: str) -> User:
    r = await call_talentlms_api(f"/users/id:{id}")
    return User(**r)


async def get_users() -> list[User]:
    r = await call_talentlms_api(f"/users")
    return [User(**x) for x in r]


async def get_course_by_id(id: int) -> Course:
    r = await call_talentlms_api(f"/courses/id:{id}")
    return Course(**r)


async def get_courses() -> list[Course]:
    r = await call_talentlms_api(f"/courses")
    return [Course(**x) for x in r]


async def get_submissions_for_unit(id: int) -> list[CourseSubmission]:
    r = await call_talentlms_api(f"/getusersprogressinunits/unit_id:{id}")
    return [CourseSubmission(**x) for x in r]


async def get_user_progress_for_course(user_id: int, course_id: int) -> UserProgress:
    r = await call_talentlms_api(
        f"/getuserstatusincourse/course_id:{course_id},user_id:{user_id}"
    )
    return UserProgress(**r)


async def add_user_to_course(user_id: int, course_id: int, role: str = "learner"):
    response = await call_talentlms_api(
        f"/addusertocourse",
        method="POST",
        params={"user_id": user_id, "course_id": course_id, "role": role},
    )
    return response
