import asyncio
from enum import Enum
import logging
from typing import Any, Callable, List, Optional, Union

import click_spinner
from rich.console import Console
from rich.table import Table
import typer

from cli.internal.wappie.types import User
import cli.internal.wappie.api as wappie
import cli.internal.talentlms.api as lms

app = typer.Typer()
console = Console()
logger = logging.getLogger(__name__)


def search_student_info(search: str) -> Union[User, list[User]]:
    if search.isnumeric():
        return lms.get_user_by_id(search)


def init_users_table() -> Table:
    table = Table(show_header=True, header_style="bold blue")
    table.add_column("👋 Name")
    table.add_column("✉️  Email")
    table.add_column("🔗 TalentLMS URL")
    table.add_column("🔗 HubSpot URL")
    return table


def add_user_to_table(table: Table, user: User) -> None:
    table.add_row(
        f"{user.first_name} {user.last_name}",
        user.email,
        f"https://winc.talentlms.com/user/info/id:{user.talentlms_user_id}",
        f"https://app.hubspot.com/contacts/5149744/contact/{user.hubspot_contact_id}"
        if user.hubspot_contact_id
        else "",
    )


@app.command()
def info(id: int):
    """
    📚 View user info by ID.
    """
    table = init_users_table()
    try:
        user = asyncio.run(wappie.get_users(talentlms_user_id=id))[0]
        add_user_to_table(table, user)
    except IndexError:
        pass
    console.print(table)


SortBy = Enum("SortBy", {k: k for k, _ in User.__fields__.items()})


@app.command(name="list")
def show(sort_by: SortBy = "last_name", reverse_sort: bool = False):
    """
    📜 List all users.
    """

    with click_spinner.spinner():
        users = asyncio.run(wappie.get_users())

    table = init_users_table()
    for u in sorted(
        users, key=lambda u: getattr(u, sort_by.value), reverse=reverse_sort
    ):
        add_user_to_table(table, u)
    console.print(table)


@app.command()
def add_to_course(user_id: int, course_id: int):
    """
    ➕ Add user to course by ID.
    """
    user = asyncio.run(lms.add_user_to_course(user_id, course_id))
    console.print(user)


@app.command()
def bulk_add_to_course(file_path: str, course_id: int):
    """
    ➕ Bulk add users via file to a course.
    First create a text file (e.g. users.txt) with IDs on each line.
    Pass the filepath as the first argument.
    """
    with open(file_path) as f:
        while line := f.readline().rstrip():
            try:
                logger.debug(f"Adding {int(line)} to {course_id}")
                add_to_course(int(line), course_id)
            except:
                pass
