import asyncio
import logging
from typing import Optional
import re
from enum import Enum

import click_spinner
from rich.console import Console
from rich.table import Table
import typer

import cli.internal.wappie.api as wappie
from cli.internal.wappie.types import GradeWork


console = Console()
logger = logging.getLogger(__name__)
app = typer.Typer()


SortBy = Enum("SortBy", {k: k for k, _ in GradeWork.__fields__.items()})


@app.command(name="list")
def show(sort_by: SortBy = "submitted_on", reverse_sort: bool = False):
    """
    📜 List all assignments.
    """

    with click_spinner.spinner():
        open_grade_work = asyncio.run(wappie.get_grade_work(open=True))

    table = Table(show_header=True, header_style="bold blue")
    table.add_column("👋 Name")
    table.add_column("🏫 Course")
    table.add_column("🔗 Assignment URL")
    table.add_column("💬 Content")
    table.add_column("🕐 Submitted")
    table.add_column("🆓 NL Leert Door")

    for gw in sorted(
        open_grade_work, key=lambda x: getattr(x, sort_by.value), reverse=reverse_sort
    ):
        table.add_row(
            f"{gw.first_name} {gw.last_name}",
            gw.course_name,
            f"https://winc.talentlms.com/unit/submissions/id:{gw.talentlms_unit_id}",
            re.sub("\n(\n)+", "\n", gw.content),
            gw.submitted_on.strftime("%Y-%m-%d %H:%M"),
            str(gw.nl_leert_door),
        )

    console.print(table)
